/*
----------------------------------------------------------------------------

 Copyright (C) Sartorius Stedim Data Analytics AB 2017 -

 Use, modification and distribution are subject to the Boost Software
 License, Version 1.0. (See http://www.boost.org/LICENSE_1_0.txt)
----------------------------------------------------------------------------
*/

#include "stdafx.h"
#include "utf8util.h"
#include <string.h>
#include <stdlib.h>
#include "sqrunner.h"

/*
Print the vectors used to create the default BiPlot in SIMCA. 
The vectors will vary depending on model type and number of components.
To get the corresponding vectors from SIMCA, create the default BiPlot, right click, Create | List.
*/
void GetBiPlot(SQ_Model hModel)
{
   SQ_ModelType eModelType;
   int iNumPredictiveComponents = 0;
   int iNumXOrthogonalComponents = 0;
   int bIsOPLSModel = 0;
   int bIsPLSModel = 0;
   int bIsPCAModel = 0;

   SQ_GetModelType(hModel, &eModelType);

   if (eModelType == SQ_OPLS || eModelType == SQ_OPLS_DA || eModelType == SQ_OPLS_Class ||
      eModelType == SQ_O2PLS || eModelType == SQ_O2PLS_DA || eModelType == SQ_O2PLS_Class)
   {
      /* Some sort of OPLS models, we will need both number of predictive and orthogonal components. */
      bIsOPLSModel = 1;
      SQ_GetNumberOfPredictiveComponents(hModel, &iNumPredictiveComponents);
      SQ_GetNumberOfXOrthogonalComponents(hModel, &iNumXOrthogonalComponents);
   }
   else if (eModelType == SQ_PLS || eModelType == SQ_PLS_DA || eModelType == SQ_PLS_Class)
   {
      /* Some sort of PLS models, only one type of components. */
      bIsPLSModel = 1;
      SQ_GetNumberOfComponents(hModel, &iNumPredictiveComponents);

   }
   else if (eModelType == SQ_PCA_X || eModelType == SQ_PCA_Y || eModelType == SQ_PCA_Class)
   {
      /* Some sort of PCA models, only one type of components. */
      bIsPCAModel = 1;
      SQ_GetNumberOfComponents(hModel, &iNumPredictiveComponents);
   }

   if (iNumPredictiveComponents > 1)
   {
      /* We have more than one component. Then we are only interested in the first and second predictive component. */
      SQ_VectorData hPCorrVectorData = NULL;
      SQ_VectorData hTCorrVectorData = NULL;
      SQ_IntVector oCompVec = 0;
      SQ_BoolVector oBoolCompVec = 0;

      SQ_InitIntVector(&oCompVec, 2);
      SQ_SetDataInIntVector(oCompVec, 1, 1);
      SQ_SetDataInIntVector(oCompVec, 2, 2);

      SQ_InitBoolVector(&oBoolCompVec, 2);
      SQ_SetDataInBoolVector(oBoolCompVec, 1, SQ_True);
      SQ_SetDataInBoolVector(oBoolCompVec, 2, SQ_True);

      if (bIsOPLSModel == 1)
      {
         /* OPLS model with more than one predictive component. */
         /* Get pq(corr)[1] pq(corr)[2]*/
         SQ_GetPqCorr(hModel, &oCompVec, &hPCorrVectorData);
      }
      else if (bIsPLSModel == 1)
      {
         /* PLS model with more than one component. */
         /* Get pc(corr)[1] pc(corr)[2]*/
         SQ_GetPcCorr(hModel, &oCompVec, &hPCorrVectorData);
      }
      else if (bIsPCAModel == 1)
      {
         /* PCA model with more than one component. */
         /* Get p(corr)[1] p(corr)[2]*/
         SQ_GetPCorrelation(hModel, &oCompVec, &hPCorrVectorData);
      }

      /* t(corr)[1] */
      SQ_GetTCorrelation(hModel, &oCompVec, &oBoolCompVec, &hTCorrVectorData);

      SQRunner_PrintVectorData(hPCorrVectorData);
      SQRunner_PrintVectorData(hTCorrVectorData);
      SQ_ClearVectorData(&hTCorrVectorData);

      SQ_SetDataInIntVector(oCompVec, 1, 2);
      SQ_SetDataInIntVector(oCompVec, 2, 1);

      /* t(corr)[2] */
      SQ_GetTCorrelation(hModel, &oCompVec, &oBoolCompVec, &hTCorrVectorData);

      SQRunner_PrintVectorData(hTCorrVectorData);

      SQ_ClearIntVector(&oCompVec);
      SQ_ClearBoolVector(&oBoolCompVec);
      SQ_ClearVectorData(&hPCorrVectorData);
      SQ_ClearVectorData(&hTCorrVectorData);
   }
   else if (iNumPredictiveComponents == 1)
   {
      if (iNumXOrthogonalComponents == 0)
      {
         SQ_VectorData hPCorrVectorData = NULL;
         SQ_VectorData hTCorrVectorData = NULL;

         if (bIsOPLSModel == 1)
         {
            /* OPLS model with one predictive component and no orthogonal component. */
            /* pq(corr)[1] */
            SQ_GetPqCorr(hModel, NULL, &hPCorrVectorData);
         }
         else if (bIsPLSModel == 1)
         {
            /* PLS model with one component. */
            /* pc(corr)[1] */
            SQ_GetPcCorr(hModel, NULL, &hPCorrVectorData);
         }
         else if (bIsPCAModel == 1)
         {
            /* PCA model with one component. */
            /* p(corr)[1] */
            SQ_GetPCorrelation(hModel, NULL, &hPCorrVectorData);
         }
         /* t(corr)[1] */
         SQ_GetTCorrelation(hModel, NULL, NULL, &hTCorrVectorData);

         SQRunner_PrintVectorData(hPCorrVectorData);
         SQRunner_PrintVectorData(hTCorrVectorData);

         SQ_ClearVectorData(&hPCorrVectorData);
         SQ_ClearVectorData(&hTCorrVectorData);
      }
      else
      {
         /* OPLS model with one predictive component and at least one orthogonal component. */
         /* po(corr)[1] and to(corr)[1] vs p(corr)[1] and t(corr)[1] */
         SQ_VectorData hPoCorrVectorData = NULL;
         SQ_VectorData hToCorrVectorData = NULL;
         SQ_VectorData hPCorrVectorData = NULL;
         SQ_VectorData hTCorrVectorData = NULL;

         SQ_IntVector oCompVec = 0;
         SQ_BoolVector oBoolCompVec = 0;

         SQ_InitIntVector(&oCompVec, 1);
         SQ_SetDataInIntVector(oCompVec, 1, 1);

         /* po(corr)[1] */
         SQ_GetPoCorrelation(hModel, &oCompVec, &hPoCorrVectorData);
         /* p(corr)[1] */
         SQ_GetPCorrelation(hModel, &oCompVec, &hPCorrVectorData);

         SQ_ClearIntVector(&oCompVec);
         SQ_InitIntVector(&oCompVec, 2);
         SQ_SetDataInIntVector(oCompVec, 1, 1);
         SQ_SetDataInIntVector(oCompVec, 2, 1);

         SQ_InitBoolVector(&oBoolCompVec, 2);
         SQ_SetDataInBoolVector(oBoolCompVec, 1, SQ_True);
         SQ_SetDataInBoolVector(oBoolCompVec, 2, SQ_False);

         /* t(corr)[1] */
         SQ_GetTCorrelation(hModel, &oCompVec, &oBoolCompVec, &hTCorrVectorData);

         SQ_SetDataInBoolVector(oBoolCompVec, 1, SQ_False);
         SQ_SetDataInBoolVector(oBoolCompVec, 2, SQ_True);

         /* to(corr)[1] */
         SQ_GetToCorr(hModel, &oCompVec, &oBoolCompVec, &hToCorrVectorData);

         SQRunner_PrintVectorData(hPCorrVectorData);
         SQRunner_PrintVectorData(hPoCorrVectorData);
         SQRunner_PrintVectorData(hTCorrVectorData);
         SQRunner_PrintVectorData(hToCorrVectorData);

         SQ_ClearIntVector(&oCompVec);
         SQ_ClearBoolVector(&oBoolCompVec);

         SQ_ClearVectorData(&hPCorrVectorData);
         SQ_ClearVectorData(&hPoCorrVectorData);
         SQ_ClearVectorData(&hTCorrVectorData);
         SQ_ClearVectorData(&hToCorrVectorData);
      }
   }
}
